/*:
 * @plugindesc CTE Compatibility Patch
 * @author (Anon + copilot)
 * 
 * @help
 * This patch ensures compatibility with the plugins listed below:
 *  - NUUN Enemy Book
 *  - NRP_ClassChangeScene
 *  - VisuMZ_1_MessageCore
 *
 * Behavior:
 *  - When a non-base language is active and CTE has a translation for an enemy or class name,
 *    the plugin replaces the corresponding $dataEnemies[id].name or $dataClasses[id].name with the
 *    translated value, backing up the original name so it can be restored when returning to the base language.
 *  - On language refresh (CTE.languageRefreshHandler), after DB load, and when maps/common events load,
 *    Show Choice commands are localized (and backed up) or restored depending on the active language.
 *  - Compatibility fixes for VisuMZ MessageCore's choice-merging behaviour so CTE localizes the
 *    final/merged choices array instead of the pre-merged array.
 *
 * Notes:
 *  - Each sub-patch checks for the presence of the targeted plugin/data before acting.
 *  - If CTE is not present, the patch does nothing.
 *
 * No plugin parameters.
 */

(function() {
    "use strict";

    // ---------- Helpers ----------
    function hasCTE() {
        return (typeof CTE !== "undefined" && CTE !== null && typeof CTE.getDataForLanguage === "function");
    }

    function isBaseLanguage() {
        try {
            return (typeof ConfigManager !== "undefined" && typeof ConfigManager.isBaseLanguage === "function")
                ? ConfigManager.isBaseLanguage()
                : false;
        } catch (e) {
            return false;
        }
    }

    function cteWarn(msg, level) {
        if (hasCTE() && typeof CTE.warn === "function") {
            CTE.warn(msg, level || 1);
        } else {
            // Fallback minimal logging to console in dev builds
            if (typeof console !== "undefined" && console.warn) console.warn("CTE Compatibility Patch: " + msg);
        }
    }

    // Safe wrapper to set/get backups via CTE if available.
    function safeSetBackup(key, field, value) {
        if (hasCTE() && typeof CTE.setBackupData === "function") {
            try { CTE.setBackupData(key, field, value); } catch (e) { /* ignore */ }
        }
    }
    function safeGetBackup(key, field) {
        if (hasCTE() && typeof CTE.getBackupData === "function") {
            try { return CTE.getBackupData(key, field); } catch (e) { return null; }
        }
        return null;
    }

    // ---------- Enemy Name Sync (NUUN Enemy Book compatibility) ----------
    function updateEnemyNamesFromCTE() {
        if (!hasCTE()) return;
        if (typeof $dataEnemies === "undefined" || !$dataEnemies) {
            // retry later if DB isn't ready yet
            setTimeout(updateEnemyNamesFromCTE, 150);
            return;
        }

        var language = (ConfigManager && typeof ConfigManager.getLanguage === "function") ? ConfigManager.getLanguage() : "";
        var base = isBaseLanguage();

        for (var id = 1; id < $dataEnemies.length; id++) {
            var enemy = $dataEnemies[id];
            if (!enemy) continue;

            var key = "enemies-" + id;

            // Non-base language & CTE has translation for this enemy name -> replace and backup
            if (!base && hasCTE() && typeof CTE.dataExistsForLanguage === "function" && CTE.dataExistsForLanguage(language, key, "name")) {
                // backup original once
                safeSetBackup(key, "name", enemy.name);
                var translated = CTE.getDataForLanguage(language, key, "name");
                if (typeof translated === "string" && translated !== "" && enemy.name !== translated) {
                    enemy.name = translated;
                }
            } else {
                // restore base name if backed up
                var backup = safeGetBackup(key, "name");
                if (backup !== null && backup !== undefined && enemy.name !== backup) {
                    enemy.name = backup;
                }
            }
        }
    }

    // ---------- Class Name Sync (NRP_ClassChangeScene compatibility) ----------
    // This mirrors the enemy approach but targets classes ($dataClasses) and category "classes-<id>".
    function updateClassNamesFromCTE() {
        if (!hasCTE()) return;
        if (typeof $dataClasses === "undefined" || !$dataClasses) {
            setTimeout(updateClassNamesFromCTE, 150);
            return;
        }

        var language = (ConfigManager && typeof ConfigManager.getLanguage === "function") ? ConfigManager.getLanguage() : "";
        var base = isBaseLanguage();

        for (var id = 1; id < $dataClasses.length; id++) {
            var cls = $dataClasses[id];
            if (!cls) continue;

            var key = "classes-" + id;

            if (!base && hasCTE() && typeof CTE.dataExistsForLanguage === "function" && CTE.dataExistsForLanguage(language, key, "name")) {
                safeSetBackup(key, "name", cls.name);
                var translated = CTE.getDataForLanguage(language, key, "name");
                if (typeof translated === "string" && translated !== "" && cls.name !== translated) {
                    cls.name = translated;
                }
            } else {
                var backup = safeGetBackup(key, "name");
                if (backup !== null && backup !== undefined && cls.name !== backup) {
                    cls.name = backup;
                }
            }
        }
    }

    // ---------- Central updater ----------
    function updateAllNamesFromCTE() {
        try {
            updateEnemyNamesFromCTE();
            updateClassNamesFromCTE();
            // Also apply choice localization/restore for currently-loaded maps & common events
            processAllLoadedChoicesFromCTE();
            // If other plugins need refreshing, do it here:
            refreshNRPSceneIfPresent();
        } catch (e) {
            // swallow any error to avoid breaking game flow
            cteWarn("updateAllNamesFromCTE error: " + (e && e.message ? e.message : e), 2);
        }
    }

    // ---------- NRP Scene refresh helper ----------
    function refreshNRPSceneIfPresent() {
        try {
            // Windows_SelectClasses and Windows_ClassInfo are defined by NRP_ClassChangeScene
            if (typeof Windows_SelectClasses !== "undefined" && typeof Windows_ClassInfo !== "undefined") {
                var scene = SceneManager._scene;
                if (scene && scene instanceof Scene_ClassChange) {
                    if (scene._selectWindow && typeof scene._selectWindow.refresh === "function") scene._selectWindow.refresh();
                    if (scene._infoWindow && typeof scene._infoWindow.refresh === "function") scene._infoWindow.refresh();
                }
            }
        } catch (e) {
            // ignore errors
        }
    }

    // ---------- Choice localization helpers ----------
    // We'll maintain per-map backups (WeakMap keyed by mapData object) to safely restore base-language choices.
    if (!window.CTE_CompatibilityPatch) window.CTE_CompatibilityPatch = {};
    window.CTE_CompatibilityPatch._mapChoiceBackups = window.CTE_CompatibilityPatch._mapChoiceBackups || new WeakMap();
    window.CTE_CompatibilityPatch.debug = window.CTE_CompatibilityPatch.debug || false;

    function ensureMapBackups(mapData) {
        var wm = window.CTE_CompatibilityPatch._mapChoiceBackups;
        if (!wm.has(mapData)) {
            wm.set(mapData, {});
        }
        return wm.get(mapData);
    }

    function makeChoiceKey(eventId, pageIndex, cmdIndex) {
        return eventId + "|" + pageIndex + "|" + cmdIndex;
    }

    // Attempt to call CTE.localizeChoiceCommand with a prepared interpreter + index.
    function safeLocalizeChoiceCommandOnList(mapData, list, eventId, pageIndex, cmdIndex) {
        try {
            var cmd = list[cmdIndex];
            if (!cmd) return;
            // backup original choices if not already backed up
            var mapBackups = ensureMapBackups(mapData);
            var key = makeChoiceKey(eventId, pageIndex, cmdIndex);
            if (!mapBackups[key]) {
                try {
                    // Deep copy (strings expected). Fallback to slice if stringify fails.
                    var originalChoices = JSON.parse(JSON.stringify(cmd.parameters[0] || []));
                    mapBackups[key] = originalChoices;
                } catch (e) {
                    mapBackups[key] = (cmd.parameters && cmd.parameters[0]) ? (cmd.parameters[0].slice ? cmd.parameters[0].slice() : null) : null;
                }
            }

            if (typeof CTE.localizeChoiceCommand === "function") {
                // Create an interpreter set up with this command list so CTE.localizeChoiceCommand can run as it expects.
                var interp = new Game_Interpreter();
                try {
                    // Game_Interpreter.setup(list, eventId) will set _list and _eventId properly
                    interp.setup(list, eventId || 0);
                } catch (e) {
                    // fallback: assign list and index directly
                    interp._list = list;
                    interp._eventId = eventId || 0;
                }
                // ensure index is set
                interp._index = cmdIndex;
                // call CTE's localization routine (defensive: try common signature)
                try {
                    CTE.localizeChoiceCommand(interp, cmdIndex);
                } catch (e1) {
                    // attempt alternate signatures if the simple call fails
                    try {
                        var command = interp._list && interp._list[cmdIndex] ? interp._list[cmdIndex] : null;
                        CTE.localizeChoiceCommand(interp, cmdIndex, command);
                    } catch (e2) {
                        try {
                            CTE.localizeChoiceCommand(command);
                        } catch (e3) {
                            // give up but don't throw
                            if (window.CTE_CompatibilityPatch.debug) {
                                cteWarn("All tried CTE.localizeChoiceCommand signatures failed for event " + eventId + " page " + pageIndex + " cmd " + cmdIndex, 1);
                            }
                        }
                    }
                }

                if (window.CTE_CompatibilityPatch.debug) {
                    cteWarn("CTE.localizeChoiceCommand called for event " + eventId + " page " + pageIndex + " cmd " + cmdIndex, 1);
                }
            } else {
                // If CTE doesn't expose localizeChoiceCommand, skip: can't localize.
                if (window.CTE_CompatibilityPatch.debug) {
                    cteWarn("CTE.localizeChoiceCommand not available; skipping localization for choices", 1);
                }
            }
        } catch (e) {
            cteWarn("safeLocalizeChoiceCommandOnList error: " + (e && e.message ? e.message : e), 1);
        }
    }

    function safeRestoreChoiceCommandOnList(mapData, list, eventId, pageIndex, cmdIndex) {
        try {
            var cmd = list[cmdIndex];
            if (!cmd) return;
            var mapBackups = ensureMapBackups(mapData);
            var key = makeChoiceKey(eventId, pageIndex, cmdIndex);
            if (mapBackups && mapBackups[key] !== undefined && mapBackups[key] !== null) {
                // Restore the choices array to the original backup (deep copy)
                try {
                    cmd.parameters[0] = JSON.parse(JSON.stringify(mapBackups[key]));
                } catch (e) {
                    cmd.parameters[0] = mapBackups[key] && mapBackups[key].slice ? mapBackups[key].slice() : mapBackups[key];
                }
                // Remove backup (we only need it until base language)
                delete mapBackups[key];
                if (window.CTE_CompatibilityPatch.debug) {
                    cteWarn("Restored choices for event " + eventId + " page " + pageIndex + " cmd " + cmdIndex, 1);
                }
            } else {
                // No backup found — nothing to restore.
                if (window.CTE_CompatibilityPatch.debug) {
                    cteWarn("No backup to restore for event " + eventId + " page " + pageIndex + " cmd " + cmdIndex, 1);
                }
            }
        } catch (e) {
            cteWarn("safeRestoreChoiceCommandOnList error: " + (e && e.message ? e.message : e), 1);
        }
    }

    // Process a single map data object's choice commands: localize or restore depending on active language.
    function processChoiceCommandsInMap(mapData) {
        if (!mapData || !mapData.events) return;
        var base = isBaseLanguage();

        for (var ei = 0; ei < mapData.events.length; ei++) {
            var ev = mapData.events[ei];
            if (!ev || !ev.pages) continue;
            for (var pi = 0; pi < ev.pages.length; pi++) {
                var page = ev.pages[pi];
                if (!page || !page.list) continue;
                var list = page.list;
                for (var ci = 0; ci < list.length; ci++) {
                    var cmd = list[ci];
                    if (!cmd) continue;
                    if (cmd.code === 102) { // Show Choices
                        if (!base) {
                            safeLocalizeChoiceCommandOnList(mapData, list, ev.id || 0, pi, ci);
                        } else {
                            safeRestoreChoiceCommandOnList(mapData, list, ev.id || 0, pi, ci);
                        }
                    }
                }
            }
        }
    }

    // Process $dataCommonEvents (they have a list property directly).
    function processCommonEventChoices() {
        if (!window.$dataCommonEvents) return;
        var base = isBaseLanguage();
        for (var i = 1; i < $dataCommonEvents.length; i++) {
            var ce = $dataCommonEvents[i];
            if (!ce || !ce.list) continue;
            var list = ce.list;
            for (var ci = 0; ci < list.length; ci++) {
                var cmd = list[ci];
                if (!cmd) continue;
                if (cmd.code === 102) {
                    // We use the common event object itself as the mapData key in the WeakMap
                    if (!window.CTE_CompatibilityPatch._mapChoiceBackups.has(ce)) {
                        window.CTE_CompatibilityPatch._mapChoiceBackups.set(ce, {});
                    }
                    if (!isBaseLanguage()) {
                        safeLocalizeChoiceCommandOnList(ce, list, ce.commonEventId || i, 0, ci);
                    } else {
                        safeRestoreChoiceCommandOnList(ce, list, ce.commonEventId || i, 0, ci);
                    }
                }
            }
        }
    }

    // Run over currently loaded maps and common events.
    function processAllLoadedChoicesFromCTE() {
        try {
            if (!hasCTE()) return;
            // Current $dataMap (currently-loaded map); DataManager loads map data into $dataMap.
            if (typeof $dataMap !== "undefined" && $dataMap && $dataMap.events) {
                processChoiceCommandsInMap($dataMap);
            }
            // Process common events.
            processCommonEventChoices();
        } catch (e) {
            cteWarn("processAllLoadedChoicesFromCTE error: " + (e && e.message ? e.message : e), 2);
        }
    }

    // ---------- Attach to CTE.languageRefreshHandler safely ----------
    function attachToCTERefreshOnce() {
        if (!hasCTE()) return false;

        try {
            var orig = CTE.languageRefreshHandler;
            // If not present or not a function, just set our handler.
            if (typeof orig !== "function") {
                CTE.languageRefreshHandler = function() {
                    updateAllNamesFromCTE();
                };
                CTE.languageRefreshHandler._CTE_compat_installed = true;
                return true;
            }

            // If we've already wrapped it, do nothing.
            if (orig._CTE_compat_installed) {
                return true;
            }

            // Create single wrapper that calls original then our updater.
            var wrapped = function() {
                try { orig(); } catch (e) { cteWarn("Original CTE.languageRefreshHandler threw: " + (e && e.message ? e.message : e), 2); }
                try { updateAllNamesFromCTE(); } catch (e) { cteWarn("updateAllNamesFromCTE threw: " + (e && e.message ? e.message : e), 2); }
            };
            wrapped._CTE_compat_installed = true;
            CTE.languageRefreshHandler = wrapped;
            return true;
        } catch (e) {
            return false;
        }
    }

    // ---------- VisuMZ MessageCore compatibility (choice merging) ----------
    // This section uses a combination of approaches:
    //  - If VisuMZ exposes addExtraShowChoices (the function that appends extra choices),
    //    we wrap it and localize after it runs (most robust).
    //  - We also provide a safe setupChoices wrapper that attempts to localize the interpreter
    //    entry before VisuMZ's setup runs (fallback if addExtraShowChoices wasn't available at wrap time).
    (function applyVisuMZChoiceCompatibility() {
        if (!hasCTE()) return;
        if (typeof Imported === "undefined" || !Imported || !Imported.VisuMZ_1_MessageCore) {
            // VisuMZ MessageCore not present
            return;
        }

        // Helper to call CTE.localizeChoiceCommand defensively.
        function callLocalizeIfAvailable(interpreter, index) {
            try {
                if (typeof CTE.localizeChoiceCommand === "function") {
                    CTE.localizeChoiceCommand(interpreter, index);
                    return true;
                }
            } catch (e) {
                // attempt alternate signature silently
                try {
                    var cmd = interpreter && interpreter._list ? interpreter._list[index] : null;
                    CTE.localizeChoiceCommand(interpreter, index, cmd);
                    return true;
                } catch (e2) {
                    try {
                        CTE.localizeChoiceCommand(cmd);
                        return true;
                    } catch (e3) {
                        // give up
                    }
                }
            }
            return false;
        }

        // Wrap addExtraShowChoices if available (VisuMZ uses this to append choices).
        try {
            if (typeof Game_Interpreter !== "undefined" && typeof Game_Interpreter.prototype.addExtraShowChoices === "function") {
                if (!CTE._visu_addExtraShowChoices_backup) {
                    CTE._visu_addExtraShowChoices_backup = Game_Interpreter.prototype.addExtraShowChoices;
                }
                if (!CTE._visu_addExtraShowChoices_wrapped) {
                    Game_Interpreter.prototype.addExtraShowChoices = function() {
                        // call original to let Visu merge into the original Show Choice command
                        CTE._visu_addExtraShowChoices_backup.apply(this, arguments);
                        // try to localize the merged Show Choice command - originalIndex often passed as second arg
                        try {
                            var originalIndex = (arguments.length >= 2 && typeof arguments[1] === "number") ? arguments[1] : this._index;
                            // ensure the command exists and is Show Choices (102)
                            if (this._list && this._list[originalIndex] && this._list[originalIndex].code === 102) {
                                callLocalizeIfAvailable(this, originalIndex);
                            }
                        } catch (e) {
                            cteWarn("VisuMZ compat: error after addExtraShowChoices: " + (e && e.message ? e.message : e), 1);
                        }
                    };
                    CTE._visu_addExtraShowChoices_wrapped = true;
                    cteWarn("CTE: Wrapped Game_Interpreter.addExtraShowChoices for VisuMZ compatibility.", 1);
                }
            }
        } catch (e) {
            cteWarn("VisuMZ compat: failed to wrap addExtraShowChoices: " + (e && e.message ? e.message : e), 1);
        }

        // As a fallback / extra safety, wrap setupChoices so that before Visu sets up the choice window
        // we attempt to localize the current interpreter list entry (this._index) if it's a Show Choices command.
        try {
            if (typeof Game_Interpreter !== "undefined" && typeof Game_Interpreter.prototype.setupChoices === "function" && !Game_Interpreter.prototype._CTECompat_wrapped_setupChoices) {
                var _orig_setupChoices = Game_Interpreter.prototype.setupChoices;
                Game_Interpreter.prototype.setupChoices = function(params) {
                    try {
                        // if the current list entry is a Show Choices command, try to localize it now.
                        if (this._list && typeof this._index === "number" && this._list[this._index] && this._list[this._index].code === 102) {
                            // localize before original runs so the UI will use translated choices
                            callLocalizeIfAvailable(this, this._index);
                        }
                    } catch (e) {
                        // ignore
                    }
                    // call original (VisuMZ's or base) setupChoices
                    return _orig_setupChoices.apply(this, arguments);
                };
                Game_Interpreter.prototype._CTECompat_wrapped_setupChoices = true;
                cteWarn("CTE: Wrapped Game_Interpreter.setupChoices for VisuMZ compatibility (fallback).", 1);
            }
        } catch (e) {
            cteWarn("VisuMZ compat: failed to wrap setupChoices: " + (e && e.message ? e.message : e), 1);
        }

    })();

    // ---------- Try to attach/wrap CTE refresh and apply an initial scan ----------
    (function tryAttachAndInitial() {
        var retries = 12;
        var attempt = function() {
            if (hasCTE()) {
                attachToCTERefreshOnce();
                // Apply an initial update (language may already be set)
                updateAllNamesFromCTE();
                return;
            }
            if (--retries > 0) setTimeout(attempt, 200);
        };
        attempt();
    })();

    // ---------- Also update after DB loads (DataManager.onLoad) ----------
    var _CTEComp_DataManager_onLoad = DataManager.onLoad;
    DataManager.onLoad = function(object) {
        _CTEComp_DataManager_onLoad.apply(this, arguments);
        try {
            var dbLoaded = typeof DataManager.isDatabaseLoaded === "function" ? DataManager.isDatabaseLoaded() : false;
            if (object === $dataEnemies || object === $dataClasses || dbLoaded) {
                // small timeout so other onLoad handlers can finish
                setTimeout(updateAllNamesFromCTE, 50);
            }
            // If a map has been loaded into object (map file), process its Show Choice commands.
            setTimeout(function() {
                try {
                    if (object && object.events) {
                        // process the loaded map (or generic object with events)
                        processChoiceCommandsInMap(object);
                    } else if (object === $dataCommonEvents) {
                        // update common events choices
                        processCommonEventChoices();
                    } else {
                        // Always safe: re-process currently-loaded map & common events after any DB load.
                        processAllLoadedChoicesFromCTE();
                    }
                } catch (e) {
                    // ignore errors
                }
            }, 60);
        } catch (e) {
            // ignore
        }
    };

    // ---------- Safety: also expose a manual refresh API ----------
    // In case some other plugin needs to trigger this explicitly.
    window.CTE_CompatibilityPatch = window.CTE_CompatibilityPatch || {};
    window.CTE_CompatibilityPatch.updateAllNamesFromCTE = updateAllNamesFromCTE;
    window.CTE_CompatibilityPatch.processAllLoadedChoicesFromCTE = processAllLoadedChoicesFromCTE;
    window.CTE_CompatibilityPatch.processChoiceCommandsInMap = processChoiceCommandsInMap;
    window.CTE_CompatibilityPatch.processCommonEventChoices = processCommonEventChoices;

    // Inform debug console that patch loaded (non-invasive)
    cteWarn("CTE Compatibility Patch loaded (updated).", 1);

})();